#!/bin/bash -

# run as user

###############################################################################
# actTexlive.sh — activate a TeX Live release already installation
#
# SYNOPSIS
#   source actTexlive.sh <YEAR>
#
# DESCRIPTION
#   This script switches to given texlive release 
#   provided it is installed for the current platform 
#   by adapting the user PATH. 
#   Thus it must be invoked by user (not by root) via `source`. 
#   It assumes that some release of TeX Live is installed and in particular the path is set accordingly. 
#   The releases must be classically installed, to be more precise, 
#   one with package manager `tlmgr`. 
#
#   TeX Live is released once a year in may 
#   and each release is identified by its year of publication <YEAR> represented by 4 digits. 
#   This script expects the TeX Live installation of releases <YEAR> are all in 
#
#       <ROOT>/<YEAR>
#
#   Where <ROOT> is by default (see https://www.tug.org/texlive/doc/texlive-en/texlive-en.html, Section 3.2.3)
#
#       /usr/local/texlive   (unixoid OS)
#       c:/texlive           (windows)
#
#   The executables are collected in 
#
#       <ROOT>/<YEAR>/bin
#
#   Prevalently, the executables depend on the TeX Live platform 
#   which is typically a combination of architecture and operating system 
#   like `x86_64-linux`. 
#   Thus they are collected in a subfolder named after the platform `<platform>` like 
#
#       <ROOT>/<YEAR>/bin/<platform>
#
#   If the preconditions are satisfied, all elements of the form 
#
#       <ROOT>/
#
#   are removed from the PATH and then 
#
#       <ROOT>/<YEAR>/bin/<platform>
#
#   is added as first element. 
#   Thus PATH contains a unique pointer to a TeX Live installation fitting the curent version and platform. 
#
#   Please use `instTexlive.sh` to install a missing release or plattform and to add further packages. 
#
#   Note that the package manager `tlmgr` is among the executables and so the following is available to the user: 
#   - `tlmgr --list` lists the packages and marks the installed ones by `i`. 
#   - `tlmgr update --list` lists the packages for which an update is available. 
#   - `tlmgr info <package>` lists various pieces of information for <package>; see `tlmgr bug <package>`
#   - `tlmgr bug <package>` lists bug contact for <package>
#
# REQUIREMENTS
#   - classical installation of TeX Live with `tlmgr` in proper place. 
#   - Must not be executed as root.
#
# ACTIONS PERFORMED
#   PATH cleanup and activation of the selected version fitting the current platform.
#
# EXIT CODES
#   0  success
#   1  problem with invocation
#      - invoker may be all but root, 
#      - invalid arguments (exactly one argument: release year shall be given by 4 digits)
#      - `tlmgr` not found 
#   2  problem with current installation 
#      - `tlmgr` not found: seems no TeX Live installed, at least no classical one. 
#      - Binaries like `tlmgr` must be in a folder with path <ROOT>/<YEAR>/bin/<platform>
#        so 4 digit <YEAR> is mandatory and so is `bin`. 
#   3  release <YEAR> is not installed  (invoke `instTexlive.sh <YEAR>` before)
#   4  current platform (given by architecture and os) not installed (maybe not supported) 
#      although the given release is installed as such. 
#
# EXAMPLES
#   switch to TeX Live 2025:
#       source ./actTexlive.sh 2025
#
# SEE ALSO
#   The sibling script is `instTexlive.sh` for installation. 
#
# AUTHOR
#   Your Name <ernst.reissner@simuline.eu>
#
###############################################################################


# messaging 
error() { printf "\e[91m✖ ERROR: %s\e[0m\n" "$1" >&2; }
warn()  { printf "\e[93m⚠ WARN: %s\e[0m\n" "$1"; }
info()  { printf "\e[94mℹ INFO: %s\e[0m\n" "$1"; }
succ()  { printf "\e[92m✔ DONE: %s\e[0m\n" "$1"; }

# check root and parameter 
[ "$EUID" -ne 0 ] || { error "Please run as user, not as root."; exit 1; }
[ "$#" -eq 1 ] || { error "Usage: $0 <year in 4 digits>."; exit 1; }
YEAR="$1"
[[ "$YEAR" =~ ^[0-9]{4}$ ]] || { error "Invalid argument: year must be exactly 4 digits."; exit 1; }

# Determine ROOT folder for all texlive installations 
TLMGR_PATH=$(command -v tlmgr) || { error "Expected 'tlmgr' but not found; classical TeX Live installed?"; exit 2; }
# Here, TLMGR_PATH=<ROOT/<YEAR>/bin/<platform>/tlmgr
BIN_DIR=$(dirname "$TLMGR_PATH")                     # <ROOT>/<YEAR>/bin/<platform>
TL_PLATFORM=$(basename "$BIN_DIR")                   # <platform>
BIN_LIT=$(basename "$(dirname "$TLMGR_PATH")")
[ "$BIN_LIT" = "bin" ] || { error "Found 'tlmgr' not in ./bin/.../. Aborting."; exit 2; }
YEAR_DIR=$(dirname "$(dirname "$BIN_DIR")")
YEAR_ORG=$(basename "$YEAR_DIR")
[[ "$YEAR_ORG" =~ ^[0-9]{4}$ ]] || { error "Invalid argument: year must be exactly 4 digits."; exit 2; }
ROOT=$(dirname "$YEAR_DIR") # <ROOT>

# each release in the folder of its year 
INSTALL_DIR="$ROOT/$YEAR"
# The subfolder of the installation with the binaries for the given architecture 
BIN_DIR="$INSTALL_DIR/bin/$TL_PLATFORM"
# Here, BIN_DIR is the new one overwriting the original one 

[ -d "$INSTALL_DIR" ] || { error "Texlive $YEAR not (properly) installed."; exit 3; }
[ -d "$BIN_DIR" ] || { error "Texlive $YEAR is not installed for your platform $TL_PLATFORM."; exit 4; }

info "switching to path including $BIN_DIR..."
# switch the path to $YEAR
OLD_IFS="$IFS"
IFS=":"
read -ra ADDR <<< "$PATH"
NEW_PATH="$BIN_DIR"
for p in "${ADDR[@]}"; do
  # match else append 
  [[ "$p" =~ ^$ROOT ]] || NEW_PATH="$NEW_PATH:$p"
done
IFS="$OLD_IFS"

export PATH="$NEW_PATH"
info "switching done"

